/*
-------------------------------------------------------------------------------

TransformMatrix.as

Matrix helper for Flash MX drawing functions

Christopher Thilgen
Macromedia
January 16, 2002

-------------------------------------------------------------------------------

General formula for matrix multiplication:

 _           _       _              _
|             |     |                |
|  a   b   c  |     |  a'   b'   c'  |
|             |     |                |
|  d   e   f  |  X  |  d'   e'   f'  |
|             |     |                |
|  g   h   i  |     |  g'   h'   i'  |
|_           _|     |_              _|

 _                                                                                  _
|                                                                                    |
|  a * a' + b * d' + c * g'    a * b' + b * e' + c * h'    a * c' + b * f' + c * i'  |
|                                                                                    |
|  d * a' + e * d' + f * g'    d * b' + e * e' + f * h'    d * c' + e * f' + f * i'  |
|                                                                                    |
|  g * a' + h * d' + i * g'    g * b' + h * e' + i * h'    g * c' + h * f' + i * i'  |
|_                                                                                  _|

-------------------------------------------------------------------------------

Scale Matrix

 _           _
|             |
|  sx  0   0  |
|             |
|  0   sy  0  |
|             |
|  0   0   1  |
|_           _|


sx is the amount of horizontal scale
sy is the amount of vertical scale

-------------------------------------------------------------------------------

Translate Matrix

 _           _
|             |
|  1   0   0  |
|             |
|  0   1   0  |
|             |
|  tx  ty  1  |
|_           _|


tx is the amount of horizontal translation
ty is the amount of vertical translation

-------------------------------------------------------------------------------

Rotate Matrix

 _                   _
|                     |
|  cos(r)  sin(r)  0  |
|                     |
|  -sin(r) cos(r)  0  |
|                     |
|  0       0       1  |
|_                   _|

r is the amount of rotation in radians

-------------------------------------------------------------------------------

The order of transformations is important...and cumulative. Doing a scale then 
a translation can result in a different outcome than the same translation 
and then a scale.

All operations are done relative to the origin of the stage. So a rotation is 
always performed about the origin.

-------------------------------------------------------------------------------

The TransformMatrix object below uses the following definition of a matrix:

 -           -
|  a   b   c  |
|  d   e   f  |
|  g   h   i  |
 -           -

This class may be used to simplify the process of building matrices to pass to
the Flash MX drawing functions.

Example:

#include "TransformMatrix.as"

_root.createEmptyMovieClip( "grad", 1 );
with ( _root.grad )
{
colors = [ 0xFF0000, 0x0000FF ];
alphas = [ 100, 100 ];
ratios = [ 0, 0xFF ];

matrix = new TransformMatrix();
matrix.scale(200,200);
matrix.translate( 200, 200 );

beginGradientFill( "linear", colors, alphas, ratios, matrix );

moveto(100,100);
lineto(100,300);
lineto(300,300);
lineto(300,100);
lineto(100,100);

endFill();
}

*/

function TransformMatrix()
{	
	this.empty = true;
}

TransformMatrix.prototype.isEmpty = function()
{
	return this.empty;
}

TransformMatrix.prototype.init = function(a,b,d,e,g,h)
{
	this.a = a;
	this.d = d;
	this.g = g;
	this.b = b;
	this.e = e;
	this.h = h;
	this.c = 0;
	this.f = 0;
	this.i = 1;
	this.empty = false;
}

TransformMatrix.prototype.concat = function(m)
{
	if ( this.isEmpty() )
	{
		this.a = m.a;
		this.d = m.d;
		this.g = m.g;
		this.b = m.b;
		this.e = m.e;
		this.h = m.h;
		this.c = m.c;
		this.f = m.f;
		this.i = m.i;
	}
	else
	{
		var result = new TransformMatrix;
		result.a = this.a * m.a + this.b * m.d + this.c * m.g;
		result.b = this.a * m.b + this.b * m.e + this.c * m.h;
		result.c = this.a * m.c + this.b * m.f + this.c * m.i;
		result.d = this.d * m.a + this.e * m.d + this.f * m.g;
		result.e = this.d * m.b + this.e * m.e + this.f * m.h;
		result.f = this.d * m.c + this.e * m.f + this.f * m.i;
		result.g = this.g * m.a + this.h * m.d + this.i * m.g;
		result.h = this.g * m.b + this.h * m.e + this.i * m.h;
		result.i = this.g * m.c + this.h * m.f + this.i * m.i;
		this.a = result.a;
		this.d = result.d;
		this.g = result.g;
		this.b = result.b;
		this.e = result.e;
		this.h = result.h;
		this.c = result.c;
		this.f = result.f;
		this.i = result.i;
	}
	this.empty = false;
}

TransformMatrix.prototype.scale = function(sx,sy)
{
	var m = new TransformMatrix;
	m.init( sx, 0, 0, sy, 0, 0 );
	return( this.concat(m) );
}

TransformMatrix.prototype.translate = function(tx,ty)
{
	var m = new TransformMatrix;
	m.init( 1, 0, 0, 1, tx, ty );
	return ( this.concat(m) );
}

TransformMatrix.prototype.rotate = function(r)
{
	// r is in degrees - must convert to radians
	var rad = ( r / 180 ) * Math.PI;
	var m = new TransformMatrix;
	var cosVal = Math.cos(rad);
	var sinVal = Math.sin(rad);
	m.init( cosVal, sinVal, -sinVal, cosVal, 0, 0 );
	return( this.concat(m) );
}
